<?php

class AdminController extends Controller
{
    private function ensureAdmin(): void
    {
        $this->requireRole(['admin']);
    }

    public function index(): void
    {
        $this->ensureAdmin();

        $db = Database::getInstance()->getConnection();

        $stats = [
            'users'        => (int)$db->query('SELECT COUNT(*) FROM users')->fetchColumn(),
            'active_users' => (int)$db->query("SELECT COUNT(*) FROM users WHERE status = 'active'")->fetchColumn(),
            'posts'        => (int)$db->query('SELECT COUNT(*) FROM posts WHERE is_deleted = 0')->fetchColumn(),
            'comments'     => (int)$db->query('SELECT COUNT(*) FROM comments WHERE is_deleted = 0')->fetchColumn(),
            'messages'     => (int)$db->query('SELECT COUNT(*) FROM messages')->fetchColumn(),
        ];

        $this->view('admin/index', [
            'title' => 'Admin Dashboard',
            'stats' => $stats,
        ]);
    }

    public function users(): void
    {
        $this->ensureAdmin();

        $db = Database::getInstance()->getConnection();
        $stmt = $db->query('SELECT u.*, p.full_name, p.institution 
            FROM users u 
            LEFT JOIN profiles p ON p.user_id = u.id
            ORDER BY u.created_at DESC');
        $users = $stmt->fetchAll();

        $this->view('admin/users', [
            'title' => 'User Management',
            'users' => $users,
        ]);
    }

    public function toggleUserStatus(int $id): void
    {
        $this->ensureAdmin();

        if (!$this->isPost() || !$this->verifyCsrf()) {
            http_response_code(400);
            echo 'Bad request';
            return;
        }

        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("UPDATE users 
            SET status = CASE WHEN status = 'active' THEN 'suspended' ELSE 'active' END
            WHERE id = :id");
        $stmt->execute(['id' => $id]);

        $this->redirect('/admin/users');
    }

    public function announcements(): void
    {
        $this->ensureAdmin();

        $model  = new Announcement();
        $items  = $model->allPaginated(50, 0);

        $this->view('admin/announcements', [
            'title'         => 'Manage Announcements',
            'announcements' => $items,
        ]);
    }

    public function newAnnouncement(): void
    {
        $this->ensureAdmin();

        if ($this->isPost()) {
            if (!$this->verifyCsrf()) {
                http_response_code(400);
                echo 'Invalid CSRF token';
                return;
            }

            $title   = trim($_POST['title'] ?? '');
            $body    = trim($_POST['body'] ?? '');
            $pinned  = !empty($_POST['pinned']);
            $active  = !empty($_POST['is_active']);
            $starts  = $_POST['starts_at'] ?: null;
            $ends    = $_POST['ends_at'] ?: null;

            $errors = [];
            if ($title === '' || $body === '') {
                $errors[] = 'Title and body are required.';
            }

            if (empty($errors)) {
                $model = new Announcement();
                $announcementId = $model->create((int)$_SESSION['user']['id'], $title, $body, $pinned, $starts, $ends);

                // Handle media uploads
                if (!empty($_FILES['media'])) {
                    $this->handleAnnouncementMediaUpload($announcementId);
                }

                // Notifications for pinned announcements could be broadcast later (e.g. via cron).

                $this->redirect('/admin/announcements');
                return;
            }

            $this->view('admin/announcement_form', [
                'title'  => 'New Announcement',
                'errors' => $errors,
            ]);
            return;
        }

        $this->view('admin/announcement_form', [
            'title' => 'New Announcement',
        ]);
    }

    private function handleAnnouncementMediaUpload(int $announcementId): int
    {
        $uploadedCount = 0;
        $maxSize = 50 * 1024 * 1024; // 50MB
        $allowedImages = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        $allowedVideos = ['video/mp4', 'video/mpeg', 'video/quicktime', 'video/x-msvideo'];
        $allowedTypes = array_merge($allowedImages, $allowedVideos);

        if (!is_array($_FILES['media']['name'])) {
            $_FILES['media']['name'] = [$_FILES['media']['name'] ?? ''];
            $_FILES['media']['size'] = [$_FILES['media']['size'] ?? 0];
            $_FILES['media']['error'] = [$_FILES['media']['error'] ?? UPLOAD_ERR_NO_FILE];
            $_FILES['media']['tmp_name'] = [$_FILES['media']['tmp_name'] ?? ''];
            $_FILES['media']['type'] = [$_FILES['media']['type'] ?? ''];
        }

        $model = new Announcement();
        $uploadOrder = 0;

        foreach ($_FILES['media']['name'] as $index => $filename) {
            if (empty($filename) || $_FILES['media']['error'][$index] !== UPLOAD_ERR_OK) {
                continue;
            }

            $fileSize = $_FILES['media']['size'][$index];
            $fileTmp = $_FILES['media']['tmp_name'][$index];
            $mimeType = $_FILES['media']['type'][$index];

            // Validate file
            if ($fileSize > $maxSize) {
                continue;
            }
            if (!in_array($mimeType, $allowedTypes)) {
                continue;
            }

            // Determine media type
            $mediaType = in_array($mimeType, $allowedImages) ? 'image' : 'video';

            // Generate unique filename
            $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
            $newFilename = 'announcement_' . $announcementId . '_' . time() . '_' . rand(1000, 9999) . '.' . $ext;
            $uploadPath = __DIR__ . '/../../public/uploads/announcements/' . $newFilename;

            if (move_uploaded_file($fileTmp, $uploadPath)) {
                $model->attachMedia($announcementId, $mediaType, $newFilename, '/uploads/announcements/' . $newFilename, $mimeType, $fileSize, $uploadOrder);
                $uploadedCount++;
                $uploadOrder++;
            }
        }

        return $uploadedCount;
    }
}

