<?php

class PostsController extends Controller
{
    public function index(): void
    {
        $postModel = new Post();
        $page      = max(1, (int)($_GET['page'] ?? 1));
        $limit     = 10;
        $offset    = ($page - 1) * $limit;
        $posts     = $postModel->getRecent($limit, $offset);

        $this->view('posts/index', [
            'title' => 'Community Posts',
            'posts' => $posts,
            'page'  => $page,
        ]);
    }

    public function show(int $id): void
    {
        $postModel    = new Post();
        $commentModel = new Comment();

        $post = $postModel->findById($id);
        if (!$post) {
            http_response_code(404);
            echo 'Post not found';
            return;
        }

        $comments = $commentModel->getByPost($id);

        $this->view('posts/show', [
            'title'    => $post['title'],
            'post'     => $post,
            'comments' => $comments,
        ]);
    }

    public function create(): void
    {
        if (empty($_SESSION['user'])) {
            $this->redirect('/auth/login');
        }

        if ($this->isPost()) {
            if (!$this->verifyCsrf()) {
                http_response_code(400);
                echo 'Invalid CSRF token';
                return;
            }

            $title = trim($_POST['title'] ?? '');
            $body  = trim($_POST['body'] ?? '');

            $errors = [];
            if ($title === '' || $body === '') {
                $errors[] = 'Title and body are required.';
            }

            if (empty($errors)) {
                $postModel = new Post();
                $postId    = $postModel->create((int)$_SESSION['user']['id'], $title, $body);

                // Handle media uploads
                if (!empty($_FILES['media'])) {
                    $uploadedCount = $this->handlePostMediaUpload($postId);
                }

                $this->redirect('/posts/show/' . $postId);
                return;
            }

            $this->view('posts/create', [
                'title'  => 'New Post',
                'errors' => $errors,
            ]);
            return;
        }

        $this->view('posts/create', [
            'title' => 'New Post',
        ]);
    }

    public function comment(int $postId): void
    {
        if (empty($_SESSION['user'])) {
            $this->redirect('/auth/login');
        }

        if (!$this->isPost() || !$this->verifyCsrf()) {
            http_response_code(400);
            echo 'Bad request';
            return;
        }

        $body = trim($_POST['body'] ?? '');
        if ($body === '') {
            $this->redirect('/posts/show/' . $postId);
            return;
        }

        $commentModel     = new Comment();
        $notificationModel = new Notification();
        $postModel        = new Post();

        $post = $postModel->findById($postId);
        if ($post) {
            $commentModel->create($postId, (int)$_SESSION['user']['id'], $body);

            // Notify post owner if not same user
            if ((int)$post['user_id'] !== (int)$_SESSION['user']['id']) {
                $notificationModel->create(
                    (int)$post['user_id'],
                    'post_comment',
                    [
                        'post_id'    => $postId,
                        'post_title' => $post['title'],
                        'from'       => $_SESSION['user']['name'] ?? $_SESSION['user']['student_id'],
                    ]
                );
            }
        }

        $this->redirect('/posts/show/' . $postId);
    }

    public function like(int $postId): void
    {
        if (empty($_SESSION['user'])) {
            $this->redirect('/auth/login');
        }

        if (!$this->isPost() || !$this->verifyCsrf()) {
            http_response_code(400);
            echo 'Bad request';
            return;
        }

        $likeModel = new Like();
        $likeModel->toggle($postId, (int)$_SESSION['user']['id']);

        $this->redirect('/posts/show/' . $postId);
    }

    public function bookmark(int $postId): void
    {
        if (empty($_SESSION['user'])) {
            $this->redirect('/auth/login');
        }

        if (!$this->isPost() || !$this->verifyCsrf()) {
            http_response_code(400);
            echo 'Bad request';
            return;
        }

        $bookmarkModel = new Bookmark();
        $bookmarkModel->togglePost((int)$_SESSION['user']['id'], $postId);

        $this->redirect('/posts/show/' . $postId);
    }

    public function repost(int $postId): void
    {
        if (empty($_SESSION['user'])) {
            $this->redirect('/auth/login');
        }

        if (!$this->isPost() || !$this->verifyCsrf()) {
            http_response_code(400);
            echo 'Bad request';
            return;
        }

        $repostModel = new Repost();
        $repostModel->toggle($postId, (int)$_SESSION['user']['id']);

        $this->redirect('/posts/show/' . $postId);
    }

    private function handlePostMediaUpload(int $postId): int
    {
        $uploadedCount = 0;
        $maxSize = 50 * 1024 * 1024; // 50MB
        $allowedImages = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        $allowedVideos = ['video/mp4', 'video/mpeg', 'video/quicktime', 'video/x-msvideo'];
        $allowedTypes = array_merge($allowedImages, $allowedVideos);

        if (!is_array($_FILES['media']['name'])) {
            $_FILES['media']['name'] = [$_FILES['media']['name'] ?? ''];
            $_FILES['media']['size'] = [$_FILES['media']['size'] ?? 0];
            $_FILES['media']['error'] = [$_FILES['media']['error'] ?? UPLOAD_ERR_NO_FILE];
            $_FILES['media']['tmp_name'] = [$_FILES['media']['tmp_name'] ?? ''];
            $_FILES['media']['type'] = [$_FILES['media']['type'] ?? ''];
        }

        $postModel = new Post();
        $uploadOrder = 0;

        foreach ($_FILES['media']['name'] as $index => $filename) {
            if (empty($filename) || $_FILES['media']['error'][$index] !== UPLOAD_ERR_OK) {
                continue;
            }

            $fileSize = $_FILES['media']['size'][$index];
            $fileTmp = $_FILES['media']['tmp_name'][$index];
            $mimeType = $_FILES['media']['type'][$index];

            // Validate file
            if ($fileSize > $maxSize) {
                continue;
            }
            if (!in_array($mimeType, $allowedTypes)) {
                continue;
            }

            // Determine media type
            $mediaType = in_array($mimeType, $allowedImages) ? 'image' : 'video';

            // Generate unique filename
            $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
            $newFilename = 'post_' . $postId . '_' . time() . '_' . rand(1000, 9999) . '.' . $ext;
            $uploadPath = __DIR__ . '/../../public/uploads/posts/' . $newFilename;

            if (move_uploaded_file($fileTmp, $uploadPath)) {
                $postModel->attachMedia($postId, $mediaType, $newFilename, '/uploads/posts/' . $newFilename, $mimeType, $fileSize, $uploadOrder);
                $uploadedCount++;
                $uploadOrder++;
            }
        }

        return $uploadedCount;
    }
}
