<?php

class ProfileController extends Controller
{
    public function index(): void
    {
        if (empty($_SESSION['user'])) {
            $this->redirect('/auth/login');
        }

        $userModel = new User();
        $user = $userModel->findById((int)$_SESSION['user']['id']);

        $this->view('profile/index', [
            'title' => 'My Profile',
            'user'  => $user,
        ]);
    }

    public function update(): void
    {
        if (empty($_SESSION['user'])) {
            $this->redirect('/auth/login');
        }

        if (!$this->isPost() || !$this->verifyCsrf()) {
            http_response_code(400);
            echo 'Bad request';
            return;
        }

        $userId = (int)$_SESSION['user']['id'];
        $fullName = trim($_POST['full_name'] ?? '');
        $institution = trim($_POST['institution'] ?? '');
        $programme = trim($_POST['programme'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $phone = trim($_POST['phone'] ?? '');
        $location = trim($_POST['location'] ?? '');
        $address = trim($_POST['address'] ?? '');
        $country = trim($_POST['country'] ?? '');

        $errors = [];
        if ($fullName === '' || $institution === '' || $programme === '') {
            $errors[] = 'Full name, institution, and programme are required.';
        }
        if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Please enter a valid email address.';
        }

        // Handle profile image upload
        if (!empty($_FILES['profile_image']['name'])) {
            $file = $_FILES['profile_image'];
            $maxSize = 5 * 1024 * 1024; // 5MB
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];

            if ($file['size'] > $maxSize) {
                $errors[] = 'Image size must not exceed 5MB.';
            } elseif (!in_array($file['type'], $allowedTypes)) {
                $errors[] = 'Only JPEG, PNG, GIF, and WebP images are allowed.';
            } elseif ($file['error'] === UPLOAD_ERR_OK) {
                // Generate unique filename
                $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                $filename = 'profile_' . $userId . '_' . time() . '.' . $ext;
                $uploadPath = __DIR__ . '/../../public/uploads/profiles/' . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
                    // Delete old profile image if exists
                    $userModel = new User();
                    $oldUser = $userModel->findById($userId);
                    if (!empty($oldUser['profile_image'])) {
                        $oldPath = __DIR__ . '/../../public/uploads/profiles/' . basename($oldUser['profile_image']);
                        if (file_exists($oldPath)) {
                            unlink($oldPath);
                        }
                    }
                    
                    // Update profile with new image
                    $userModel->updateProfileImage($userId, $filename);
                    $_SESSION['user']['profile_image'] = $filename;
                } else {
                    $errors[] = 'Failed to upload image. Please try again.';
                }
            }
        }

        if (empty($errors)) {
            $userModel = new User();
            $userModel->updateProfile($userId, $fullName, $institution, $programme, $email, $phone, $location, $address, $country);

            // Update session name
            $_SESSION['user']['name'] = $fullName;

            $success = 'Profile updated successfully!';
        }

        $userModel = new User();
        $user = $userModel->findById($userId);

        $this->view('profile/index', [
            'title'   => 'My Profile',
            'user'    => $user,
            'success' => $success ?? null,
            'errors'  => $errors,
        ]);
    }

    public function changePassword(): void
    {
        if (empty($_SESSION['user'])) {
            $this->redirect('/auth/login');
        }

        if (!$this->isPost() || !$this->verifyCsrf()) {
            http_response_code(400);
            echo 'Bad request';
            return;
        }

        $userId = (int)$_SESSION['user']['id'];
        $currentPassword = $_POST['current_password'] ?? '';
        $newPassword = $_POST['new_password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';

        $errors = [];

        $userModel = new User();
        $user = $userModel->findById($userId);

        if (!$user || !password_verify($currentPassword, $user['password_hash'])) {
            $errors[] = 'Current password is incorrect.';
        }

        if (strlen($newPassword) < 8) {
            $errors[] = 'New password must be at least 8 characters.';
        }

        if ($newPassword !== $confirmPassword) {
            $errors[] = 'Passwords do not match.';
        }

        if ($newPassword === $currentPassword) {
            $errors[] = 'New password must be different from current password.';
        }

        if (empty($errors)) {
            $newHash = password_hash($newPassword, PASSWORD_BCRYPT);
            $userModel->updatePassword($userId, $newHash);
            $success = 'Password changed successfully!';
        }

        $user = $userModel->findById($userId);

        $this->view('profile/index', [
            'title'        => 'My Profile',
            'user'         => $user,
            'pwdSuccess'   => $success ?? null,
            'pwdErrors'    => $errors,
            'showPwdForm'  => true,
        ]);
    }
}
