<?php

class Announcement extends Model
{
    public function create(int $adminId, string $title, string $body, bool $pinned, ?string $startsAt, ?string $endsAt): int
    {
        $stmt = $this->db->prepare('INSERT INTO announcements 
            (admin_id, title, body, pinned, is_active, starts_at, ends_at)
            VALUES (:admin_id, :title, :body, :pinned, 1, :starts_at, :ends_at)');
        $stmt->execute([
            'admin_id'  => $adminId,
            'title'     => $title,
            'body'      => $body,
            'pinned'    => $pinned ? 1 : 0,
            'starts_at' => $startsAt,
            'ends_at'   => $endsAt,
        ]);
        return (int)$this->db->lastInsertId();
    }

    public function update(int $id, string $title, string $body, bool $pinned, bool $isActive, ?string $startsAt, ?string $endsAt): bool
    {
        $stmt = $this->db->prepare('UPDATE announcements
            SET title = :title,
                body = :body,
                pinned = :pinned,
                is_active = :is_active,
                starts_at = :starts_at,
                ends_at = :ends_at
            WHERE id = :id');
        return $stmt->execute([
            'id'        => $id,
            'title'     => $title,
            'body'      => $body,
            'pinned'    => $pinned ? 1 : 0,
            'is_active' => $isActive ? 1 : 0,
            'starts_at' => $startsAt,
            'ends_at'   => $endsAt,
        ]);
    }

    public function delete(int $id): bool
    {
        $stmt = $this->db->prepare('DELETE FROM announcements WHERE id = :id');
        return $stmt->execute(['id' => $id]);
    }

    public function findById(int $id): ?array
    {
        $stmt = $this->db->prepare('SELECT a.*, u.student_id, pr.full_name
            FROM announcements a
            JOIN users u ON u.id = a.admin_id
            LEFT JOIN profiles pr ON pr.user_id = u.id
            WHERE a.id = :id');
        $stmt->execute(['id' => $id]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function getActiveForDashboard(int $limit = 5): array
    {
        $now = date('Y-m-d H:i:s');
        $sql = 'SELECT a.*, u.student_id, pr.full_name
                FROM announcements a
                JOIN users u ON u.id = a.admin_id
                LEFT JOIN profiles pr ON pr.user_id = u.id
                WHERE a.is_active = 1
                  AND (a.starts_at IS NULL OR a.starts_at <= ?)
                  AND (a.ends_at IS NULL OR a.ends_at >= ?)
                ORDER BY a.pinned DESC, a.created_at DESC
                LIMIT ?';
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$now, $now, $limit]);
        return $stmt->fetchAll();
    }

    public function getActive(int $limit = 50, int $offset = 0): array
    {
        $now = date('Y-m-d H:i:s');
        $sql = 'SELECT a.*, u.student_id, pr.full_name
                FROM announcements a
                JOIN users u ON u.id = a.admin_id
                LEFT JOIN profiles pr ON pr.user_id = u.id
                WHERE a.is_active = 1
                  AND (a.starts_at IS NULL OR a.starts_at <= ?)
                  AND (a.ends_at IS NULL OR a.ends_at >= ?)
                ORDER BY a.pinned DESC, a.created_at DESC
                LIMIT ? OFFSET ?';
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$now, $now, $limit, $offset]);
        return $stmt->fetchAll();
    }

    public function getPinned(int $limit = 10): array
    {
        $now = date('Y-m-d H:i:s');
        $sql = 'SELECT a.*, u.student_id, pr.full_name
                FROM announcements a
                JOIN users u ON u.id = a.admin_id
                LEFT JOIN profiles pr ON pr.user_id = u.id
                WHERE a.is_active = 1
                  AND a.pinned = 1
                  AND (a.starts_at IS NULL OR a.starts_at <= ?)
                  AND (a.ends_at IS NULL OR a.ends_at >= ?)
                ORDER BY a.created_at DESC
                LIMIT ?';
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$now, $now, $limit]);
        return $stmt->fetchAll();
    }

    public function allPaginated(int $limit = 20, int $offset = 0): array
    {
        $sql = 'SELECT a.*, u.student_id, pr.full_name
                FROM announcements a
                JOIN users u ON u.id = a.admin_id
                LEFT JOIN profiles pr ON pr.user_id = u.id
                ORDER BY a.pinned DESC, a.created_at DESC
                LIMIT ? OFFSET ?';
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$limit, $offset]);
        return $stmt->fetchAll();
    }

    public function attachMedia(int $announcementId, string $mediaType, string $filename, string $filePath, string $mimeType = '', int $fileSize = 0, int $uploadOrder = 0): int
    {
        $stmt = $this->db->prepare('INSERT INTO announcement_media (announcement_id, media_type, filename, file_path, mime_type, file_size, upload_order)
            VALUES (:announcement_id, :media_type, :filename, :file_path, :mime_type, :file_size, :upload_order)');
        $stmt->execute([
            'announcement_id' => $announcementId,
            'media_type'      => $mediaType,
            'filename'        => $filename,
            'file_path'       => $filePath,
            'mime_type'       => $mimeType,
            'file_size'       => $fileSize,
            'upload_order'    => $uploadOrder,
        ]);
        return (int)$this->db->lastInsertId();
    }

    public function getAnnouncementMedia(int $announcementId): array
    {
        $stmt = $this->db->prepare('SELECT * FROM announcement_media WHERE announcement_id = :announcement_id ORDER BY upload_order ASC, created_at ASC');
        $stmt->execute(['announcement_id' => $announcementId]);
        return $stmt->fetchAll();
    }

    public function getMedia(int $announcementId): array
    {
        return $this->getAnnouncementMedia($announcementId);
    }

    public function deleteMedia(int $mediaId): bool
    {
        $stmt = $this->db->prepare('DELETE FROM announcement_media WHERE id = :id');
        return $stmt->execute(['id' => $mediaId]);
    }
}
