<?php

class Post extends Model
{
    public function create(int $userId, string $title, string $body): int
    {
        $stmt = $this->db->prepare('INSERT INTO posts (user_id, title, body) VALUES (:user_id, :title, :body)');
        $stmt->execute([
            'user_id' => $userId,
            'title'   => $title,
            'body'    => $body,
        ]);
        return (int)$this->db->lastInsertId();
    }

    public function update(int $postId, int $userId, string $title, string $body): bool
    {
        $stmt = $this->db->prepare('UPDATE posts SET title = :title, body = :body, updated_at = NOW()
            WHERE id = :id AND user_id = :user_id AND is_deleted = 0');
        return $stmt->execute([
            'id'      => $postId,
            'user_id' => $userId,
            'title'   => $title,
            'body'    => $body,
        ]);
    }

    public function softDelete(int $postId, int $userId): bool
    {
        $stmt = $this->db->prepare('UPDATE posts SET is_deleted = 1 WHERE id = :id AND user_id = :user_id');
        return $stmt->execute([
            'id'      => $postId,
            'user_id' => $userId,
        ]);
    }

    public function findById(int $id): ?array
    {
        $stmt = $this->db->prepare('SELECT p.*, u.student_id, pr.full_name, pr.profile_image
            FROM posts p
            JOIN users u ON u.id = p.user_id
            LEFT JOIN profiles pr ON pr.user_id = u.id
            WHERE p.id = :id AND p.is_deleted = 0');
        $stmt->execute(['id' => $id]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function getByIdWithEngagement(int $id): ?array
    {
        $stmt = $this->db->prepare('
            SELECT p.*, u.student_id, pr.full_name, pr.profile_image,
                   COUNT(DISTINCT c.id) as comments_count,
                   COUNT(DISTINCT l.id) as likes_count,
                   COUNT(DISTINCT r.id) as reposts_count
            FROM posts p
            JOIN users u ON u.id = p.user_id
            LEFT JOIN profiles pr ON pr.user_id = u.id
            LEFT JOIN comments c ON c.post_id = p.id AND c.is_deleted = 0
            LEFT JOIN likes l ON l.post_id = p.id
            LEFT JOIN reposts r ON r.post_id = p.id
            WHERE p.id = :id AND p.is_deleted = 0
            GROUP BY p.id
        ');
        $stmt->execute(['id' => $id]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function getRecent(int $limit = 20, int $offset = 0): array
    {
        $stmt = $this->db->prepare('SELECT p.*, u.student_id, pr.full_name, pr.profile_image,
                (SELECT COUNT(*) FROM comments c WHERE c.post_id = p.id AND c.is_deleted = 0) AS comments_count,
                (SELECT COUNT(*) FROM likes l WHERE l.post_id = p.id) AS likes_count,
                (SELECT COUNT(*) FROM reposts r WHERE r.post_id = p.id) AS reposts_count
            FROM posts p
            JOIN users u ON u.id = p.user_id
            LEFT JOIN profiles pr ON pr.user_id = u.id
            WHERE p.is_deleted = 0
            ORDER BY p.created_at DESC
            LIMIT ? OFFSET ?');
        $stmt->execute([$limit, $offset]);
        return $stmt->fetchAll();
    }

    public function getTrending(int $limit = 5): array
    {
        // Simple trending score based on recent comments + likes + reposts
        $sql = 'SELECT p.*, u.student_id, pr.full_name, pr.profile_image,
                    COUNT(DISTINCT c.id) AS comments_count,
                    COUNT(DISTINCT l.id) AS likes_count,
                    COUNT(DISTINCT r.id) AS reposts_count,
                    (COUNT(DISTINCT c.id) * 2 + COUNT(DISTINCT l.id) + COUNT(DISTINCT r.id) * 3) AS score
                FROM posts p
                JOIN users u ON u.id = p.user_id
                LEFT JOIN profiles pr ON pr.user_id = u.id
                LEFT JOIN comments c ON c.post_id = p.id AND c.is_deleted = 0 AND c.created_at >= (NOW() - INTERVAL 14 DAY)
                LEFT JOIN likes l ON l.post_id = p.id AND l.created_at >= (NOW() - INTERVAL 14 DAY)
                LEFT JOIN reposts r ON r.post_id = p.id AND r.created_at >= (NOW() - INTERVAL 14 DAY)
                WHERE p.is_deleted = 0
                GROUP BY p.id
                HAVING score > 0
                ORDER BY score DESC, p.created_at DESC
                LIMIT ?';
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$limit]);
        return $stmt->fetchAll();
    }

    public function attachMedia(int $postId, string $mediaType, string $filename, string $filePath, string $mimeType = '', int $fileSize = 0, int $uploadOrder = 0): int
    {
        $stmt = $this->db->prepare('INSERT INTO post_media (post_id, media_type, filename, file_path, mime_type, file_size, upload_order)
            VALUES (:post_id, :media_type, :filename, :file_path, :mime_type, :file_size, :upload_order)');
        $stmt->execute([
            'post_id'        => $postId,
            'media_type'     => $mediaType,
            'filename'       => $filename,
            'file_path'      => $filePath,
            'mime_type'      => $mimeType,
            'file_size'      => $fileSize,
            'upload_order'   => $uploadOrder,
        ]);
        return (int)$this->db->lastInsertId();
    }

    public function getPostMedia(int $postId): array
    {
        $stmt = $this->db->prepare('SELECT * FROM post_media WHERE post_id = :post_id ORDER BY upload_order ASC, created_at ASC');
        $stmt->execute(['post_id' => $postId]);
        return $stmt->fetchAll();
    }

    public function getByUser(int $userId): array
    {
        $stmt = $this->db->prepare('SELECT p.*, u.student_id, pr.full_name, pr.profile_image,
                (SELECT COUNT(*) FROM comments c WHERE c.post_id = p.id AND c.is_deleted = 0) AS comments_count,
                (SELECT COUNT(*) FROM likes l WHERE l.post_id = p.id) AS likes_count,
                (SELECT COUNT(*) FROM reposts r WHERE r.post_id = p.id) AS reposts_count
            FROM posts p
            JOIN users u ON u.id = p.user_id
            LEFT JOIN profiles pr ON pr.user_id = u.id
            WHERE p.user_id = :user_id AND p.is_deleted = 0
            ORDER BY p.created_at DESC');
        $stmt->execute(['user_id' => $userId]);
        return $stmt->fetchAll();
    }

    public function deleteMedia(int $mediaId): bool
    {
        $stmt = $this->db->prepare('DELETE FROM post_media WHERE id = :id');
        return $stmt->execute(['id' => $mediaId]);
    }
}
