<?php

class User extends Model
{
    public function findByStudentId(string $studentId): ?array
    {
        $stmt = $this->db->prepare('SELECT u.*, p.full_name, p.institution, p.programme, p.profile_image 
            FROM users u 
            LEFT JOIN profiles p ON p.user_id = u.id
            WHERE u.student_id = :student_id
            LIMIT 1');
        $stmt->execute(['student_id' => $studentId]);
        $user = $stmt->fetch();
        return $user ?: null;
    }

    public function createStudent(string $studentId, string $fullName, string $institution, string $programme, string $passwordHash): int
    {
        $this->db->beginTransaction();
        try {
            $stmt = $this->db->prepare('INSERT INTO users (student_id, password_hash, role, status) 
                VALUES (:student_id, :password_hash, :role, :status)');
            $stmt->execute([
                'student_id'    => $studentId,
                'password_hash' => $passwordHash,
                'role'          => 'student',
                'status'        => 'active',
            ]);

            $userId = (int)$this->db->lastInsertId();

            $stmt = $this->db->prepare('INSERT INTO profiles (user_id, full_name, institution, programme) 
                VALUES (:user_id, :full_name, :institution, :programme)');
            $stmt->execute([
                'user_id'     => $userId,
                'full_name'   => $fullName,
                'institution' => $institution,
                'programme'   => $programme,
            ]);

            $this->db->commit();
            return $userId;
        } catch (Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
    }

    public function findById(int $userId): ?array
    {
        $stmt = $this->db->prepare('SELECT u.*, p.full_name, p.institution, p.programme, p.profile_image 
            FROM users u 
            LEFT JOIN profiles p ON p.user_id = u.id
            WHERE u.id = :id
            LIMIT 1');
        $stmt->execute(['id' => $userId]);
        $user = $stmt->fetch();
        return $user ?: null;
    }

    public function updateProfile(int $userId, string $fullName, string $institution, string $programme, string $email = '', string $phone = '', string $location = '', string $address = '', string $country = ''): bool
    {
        $stmt = $this->db->prepare('UPDATE profiles SET full_name = :full_name, institution = :institution, programme = :programme, 
            email = :email, phone = :phone, location = :location, address = :address, country = :country
            WHERE user_id = :user_id');
        return $stmt->execute([
            'user_id'     => $userId,
            'full_name'   => $fullName,
            'institution' => $institution,
            'programme'   => $programme,
            'email'       => $email,
            'phone'       => $phone,
            'location'    => $location,
            'address'     => $address,
            'country'     => $country,
        ]);
    }

    public function updateProfileImage(int $userId, string $imagePath): bool
    {
        $stmt = $this->db->prepare('UPDATE profiles SET profile_image = :profile_image WHERE user_id = :user_id');
        return $stmt->execute([
            'user_id'        => $userId,
            'profile_image'  => $imagePath,
        ]);
    }

    public function updatePassword(int $userId, string $passwordHash): bool
    {
        $stmt = $this->db->prepare('UPDATE users SET password_hash = :password_hash WHERE id = :id');
        return $stmt->execute([
            'id'              => $userId,
            'password_hash'   => $passwordHash,
        ]);
    }

    public function createPasswordResetToken(int $userId): string
    {
        // Generate a unique token
        $token = bin2hex(random_bytes(32));
        
        // Delete any existing tokens for this user
        $stmt = $this->db->prepare('DELETE FROM password_reset_tokens WHERE user_id = :user_id');
        $stmt->execute(['user_id' => $userId]);
        
        // Create new reset token (valid for 1 hour)
        $stmt = $this->db->prepare('INSERT INTO password_reset_tokens (user_id, token, expires_at) 
            VALUES (:user_id, :token, DATE_ADD(NOW(), INTERVAL 1 HOUR))');
        $stmt->execute([
            'user_id' => $userId,
            'token'   => $token,
        ]);
        
        return $token;
    }

    public function validatePasswordResetToken(string $token): ?array
    {
        $stmt = $this->db->prepare('SELECT prt.*, u.id as user_id, p.email 
            FROM password_reset_tokens prt
            JOIN users u ON u.id = prt.user_id
            JOIN profiles p ON p.user_id = u.id
            WHERE prt.token = :token AND prt.expires_at > NOW()
            LIMIT 1');
        $stmt->execute(['token' => $token]);
        $result = $stmt->fetch();
        return $result ?: null;
    }

    public function resetPassword(string $token, string $passwordHash): bool
    {
        $this->db->beginTransaction();
        try {
            // Get the user from the token
            $stmt = $this->db->prepare('SELECT user_id FROM password_reset_tokens WHERE token = :token AND expires_at > NOW()');
            $stmt->execute(['token' => $token]);
            $result = $stmt->fetch();
            
            if (!$result) {
                return false;
            }
            
            $userId = (int)$result['user_id'];
            
            // Update password
            $this->updatePassword($userId, $passwordHash);
            
            // Delete the reset token
            $stmt = $this->db->prepare('DELETE FROM password_reset_tokens WHERE token = :token');
            $stmt->execute(['token' => $token]);
            
            $this->db->commit();
            return true;
        } catch (Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
}

